import { NextResponse } from 'next/server';
import { promises as fs } from 'fs';
import path from 'path';
import { getServerSession } from "next-auth/next"
import { authOptions } from '@/lib/auth';

// Define the path to the JSON file
const dataFilePath = path.join(process.cwd(), 'data/announcements.json');

interface Announcement {
    id: number;
    title: string;
    content: string;
    image?: string | null;
    author: string;
    createdAt: string;
    scheduledTime?: string | null;
}

// Helper function to read the data
async function readData(): Promise<Announcement[]> {
    try {
        await fs.access(dataFilePath);
        const jsonData = await fs.readFile(dataFilePath, 'utf-8');
        return jsonData ? JSON.parse(jsonData) : [];
    } catch (error) {
        return [];
    }
}

// GET handler to fetch visible announcements
export async function GET(req: Request) {
    try {
        const session = await getServerSession(authOptions);
        const { searchParams } = new URL(req.url);
        const includeAll = searchParams.get('include_all') === 'true';

        const announcements = await readData();
        const now = new Date();

        let visibleAnnouncements = announcements;

        // If the user is not an admin or is not requesting all announcements, filter them.
        if (!(session?.user as any)?.isAdmin || !includeAll) {
            visibleAnnouncements = announcements.filter(announcement => {
                if (!announcement.scheduledTime) {
                    return true; // Always show if not scheduled
                }
                return now >= new Date(announcement.scheduledTime); // Show if scheduled time has passed
            });
        }

        // For admins, sort all announcements. For others, sort the visible ones.
        visibleAnnouncements.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
        
        return NextResponse.json(visibleAnnouncements);
    } catch (error) {
        console.error("Failed to read announcements:", error);
        return NextResponse.json({ error: 'Failed to retrieve announcements' }, { status: 500 });
    }
}

// POST handler to create a new announcement
export async function POST(req: Request) {
    const session = await getServerSession(authOptions);

    if (!(session?.user as any)?.isAdmin) {
        return NextResponse.json({ error: 'Unauthorized' }, { status: 403 });
    }

    try {
        const { title, content, image, scheduledTime } = await req.json();

        if (!title || !content) {
            return NextResponse.json({ error: 'Title and content are required' }, { status: 400 });
        }

        const announcements = await readData();

        const newAnnouncement: Announcement = {
            id: Date.now(), // Simple unique ID
            title,
            content,
            image: image || null,
            author: session.user?.name || 'Administrator',
            createdAt: new Date().toISOString(),
            scheduledTime: scheduledTime || null,
        };

        announcements.push(newAnnouncement);

        await fs.writeFile(dataFilePath, JSON.stringify(announcements, null, 2));

        return NextResponse.json(newAnnouncement, { status: 201 });
    } catch (error) {
        console.error("Failed to create announcement:", error);
        return NextResponse.json({ error: 'Failed to create announcement' }, { status: 500 });
    }
}

// DELETE handler to remove an announcement
export async function DELETE(req: Request) {
    const session = await getServerSession(authOptions);

    if (!(session?.user as any)?.isAdmin) {
        return NextResponse.json({ error: 'Unauthorized' }, { status: 403 });
    }

    try {
        const { searchParams } = new URL(req.url);
        const idToDelete = searchParams.get('id');

        if (!idToDelete) {
            return NextResponse.json({ error: 'Announcement ID is required' }, { status: 400 });
        }

        const announcements = await readData();
        
        const filteredAnnouncements = announcements.filter(ann => ann.id !== parseInt(idToDelete, 10));

        if (announcements.length === filteredAnnouncements.length) {
            return NextResponse.json({ error: 'Announcement not found' }, { status: 404 });
        }

        await fs.writeFile(dataFilePath, JSON.stringify(filteredAnnouncements, null, 2));

        return NextResponse.json({ message: 'Announcement deleted successfully' }, { status: 200 });
    } catch (error) {
        console.error("Failed to delete announcement:", error);
        return NextResponse.json({ error: 'Failed to delete announcement' }, { status: 500 });
    }
}
