import { NextRequest, NextResponse } from 'next/server';
import { promises as fs } from 'fs';
import path from 'path';
import { getServerSession } from "next-auth/next";
import { authOptions } from '@/lib/auth';

const dataFilePath = path.join(process.cwd(), 'data/blacklist.json');

// Helper to read data
async function readBlacklist() {
    try {
        await fs.access(dataFilePath);
        const jsonData = await fs.readFile(dataFilePath, 'utf-8');
        return jsonData ? JSON.parse(jsonData) : [];
    } catch (error) {
        return [];
    }
}

// Helper to write data
async function writeBlacklist(data: any) {
    await fs.writeFile(dataFilePath, JSON.stringify(data, null, 2));
}

// GET: Fetch all blacklisted users
export async function GET() {
    const blacklist = await readBlacklist();
    return NextResponse.json(blacklist);
}

// POST: Add a user to the blacklist
export async function POST(req: NextRequest) {
    const session = await getServerSession(authOptions);
    if (!(session?.user as any)?.isAdmin) {
        return NextResponse.json({ error: 'Unauthorized' }, { status: 403 });
    }

    const { userId } = await req.json();
    if (!userId) {
        return NextResponse.json({ error: 'User ID is required' }, { status: 400 });
    }

    const blacklist = await readBlacklist();
    if (blacklist.some((user: any) => user.id === userId)) {
        return NextResponse.json({ error: 'User is already on the blacklist' }, { status: 409 });
    }

    const newUser = {
        id: userId,
        addedBy: session.user?.name,
        addedAt: new Date().toISOString(),
        userInfo: null, // Will be populated on user login
        permissions: {
            canUseSupport: false,
            canUseApplications: false,
        }
    };

    blacklist.push(newUser);
    await writeBlacklist(blacklist);

    return NextResponse.json(newUser, { status: 201 });
}

// PUT: Update user permissions or info
export async function PUT(req: NextRequest) {
    const session = await getServerSession(authOptions);
    const { userId, permissions, userInfo } = await req.json();

    if (!userId) {
        return NextResponse.json({ error: 'User ID is required' }, { status: 400 });
    }

    const blacklist = await readBlacklist();
    const userIndex = blacklist.findIndex((user: any) => user.id === userId);

    if (userIndex === -1) {
        return NextResponse.json({ error: 'User not found on blacklist' }, { status: 404 });
    }

    // If the update is for permissions, check for admin
    if (permissions && !(session?.user as any)?.isAdmin) {
         return NextResponse.json({ error: 'Unauthorized' }, { status: 403 });
    }

    // If the update is for userInfo, any authenticated user can update their own info
    if (userInfo && session?.user && (session.user as any).id === userId) {
        blacklist[userIndex].userInfo = userInfo;
    } else if (permissions) { // Admin updates permissions
        blacklist[userIndex].permissions = { ...blacklist[userIndex].permissions, ...permissions };
    } else {
        return NextResponse.json({ error: 'Invalid update request' }, { status: 400 });
    }

    await writeBlacklist(blacklist);
    return NextResponse.json(blacklist[userIndex]);
}

// DELETE: Remove a user from the blacklist
export async function DELETE(req: NextRequest) {
    const session = await getServerSession(authOptions);
    if (!(session?.user as any)?.isAdmin) {
        return NextResponse.json({ error: 'Unauthorized' }, { status: 403 });
    }

    const { userId } = await req.json();
    if (!userId) {
        return NextResponse.json({ error: 'User ID is required' }, { status: 400 });
    }

    let blacklist = await readBlacklist();
    const initialLength = blacklist.length;
    blacklist = blacklist.filter((user: any) => user.id !== userId);

    if (blacklist.length === initialLength) {
        return NextResponse.json({ error: 'User not found on blacklist' }, { status: 404 });
    }

    await writeBlacklist(blacklist);
    return NextResponse.json({ success: true });
}
