import { NextResponse } from 'next/server';
import { promises as fs } from 'fs';
import path from 'path';
import { getServerSession } from "next-auth/next";
import { authOptions } from '@/lib/auth';

const dataFilePath = path.join(process.cwd(), 'data/settings.json');

// Helper function to read settings
async function readSettings() {
    try {
        await fs.access(dataFilePath);
        const jsonData = await fs.readFile(dataFilePath, 'utf-8');
        return JSON.parse(jsonData);
    } catch (error) {
        // Return default settings if file doesn't exist or is empty
        return {
            maintenanceMode: false,
            siteTitle: "RedCity RP",
            socialLinks: { discord: "", youtube: "", facebook: "", twitter: "", instagram: "" }
        };
    }
}

// GET handler to fetch settings
export async function GET() {
    try {
        const settings = await readSettings();
        return NextResponse.json(settings);
    } catch (error) {
        console.error("Failed to read settings:", error);
        return NextResponse.json({ error: 'Failed to retrieve settings' }, { status: 500 });
    }
}

// POST handler to update settings
export async function POST(req: Request) {
    const session = await getServerSession(authOptions);

    if (!(session?.user as any)?.isAdmin) {
        return NextResponse.json({ error: 'Unauthorized' }, { status: 403 });
    }

    try {
        const newSettings = await req.json();
        // Basic validation can be added here if needed

        await fs.writeFile(dataFilePath, JSON.stringify(newSettings, null, 2));

        return NextResponse.json({ success: true, settings: newSettings });
    } catch (error) {
        console.error("Failed to update settings:", error);
        return NextResponse.json({ error: 'Failed to update settings' }, { status: 500 });
    }
}
