import { NextRequest, NextResponse } from 'next/server';

export async function POST(req: NextRequest) {
  const { message, user, severity } = await req.json();

  if (!message || !user || !severity) {
    return NextResponse.json({ error: 'Missing message, user, or severity data' }, { status: 400 });
  }

  const discordWebhookUrl = process.env.DISCORD_WEBHOOK_URL;
  const supportRoleId = process.env.DISCORD_SUPPORT_ROLE_ID;
  const adminRoleId = process.env.DISCORD_ADMIN_ROLE_ID;

  if (!discordWebhookUrl) {
    console.error('Missing DISCORD_WEBHOOK_URL environment variable');
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
  }

  let pingContent = '';
  let color = 0x808080; // Grey for basic

  switch (severity) {
    case 'średni':
      if (supportRoleId) pingContent = `<@&${supportRoleId}>`;
      color = 0xffa500; // Orange
      break;
    case 'poważny':
      if (adminRoleId) pingContent = `<@&${adminRoleId}>`;
      color = 0xff0000; // Red
      break;
  }

  const embed = {
    title: 'Nowe zgłoszenie supportu',
    color: color,
    fields: [
      {
        name: 'Użytkownik',
        value: user.name || 'Nieznany',
        inline: true,
      },
      {
        name: 'Email',
        value: user.email || 'Nieznany',
        inline: true,
      },
      {
        name: 'Waga',
        value: severity.charAt(0).toUpperCase() + severity.slice(1), // Capitalize first letter
        inline: true,
      },
      {
        name: 'Wiadomość',
        value: message,
      },
    ],
    thumbnail: {
      url: user.image || '',
    },
    timestamp: new Date().toISOString(),
  };

  const payload = {
    content: pingContent,
    embeds: [embed],
  };

  try {
    const response = await fetch(discordWebhookUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(payload),
    });

    if (response.ok) {
      return NextResponse.json({ success: true });
    } else {
      console.error('Error sending to Discord:', await response.text());
      return NextResponse.json({ error: 'Failed to send message to Discord' }, { status: 500 });
    }
  } catch (error) {
    console.error('Error sending to Discord:', error);
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 });
  }
}
