"use client";

import { useState, useEffect } from "react";
import Image from "next/image";
import Link from "next/link";
import { FaDiscord, FaTiktok, FaYoutube } from "react-icons/fa";
import { FiCopy, FiUsers, FiShoppingCart, FiAward, FiGlobe, FiChevronLeft, FiChevronRight, FiMenu, FiX } from "react-icons/fi";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import Faq from "@/components/faq";

const translations = {
  pl: {
    loading: "Ładowanie...",
    playerPanel: "Panel Gracza",
    welcome: "Witaj w",
    motto: "Twoje miasto, Twoja historia, Twoje zasady.",
    onlinePlayers: "Gracze online:",
    copyIp: "Skopiuj IP Serwera",
    copied: "Skopiowano!",
    joinDiscord: "Dołącz do Discorda",
    shopTitle: "Nasz Sklep",
    shopDescription: "W naszym sklepie znajdziesz unikalne przedmioty, pojazdy i wiele więcej, aby wzbogacić swoje doświadczenie na serwerze. Każdy zakup wspiera rozwój RedCity!",
    goToShop: "Przejdź do sklepu",
    rolesTitle: "Role i Rangi",
    rolesDescription: "Chcesz zostać częścią administracji lub dołączyć do jednej z frakcji? Zobacz dostępne role i aplikuj już dziś, aby zacząć swoją karierę w RedCity.",
    applyForRank: "Aplikuj o rangę",
    home: "Strona główna",
    ranks: "Rangi",
    support: "Support",
  },
  en: {
    loading: "Loading...",
    playerPanel: "Player Panel",
    welcome: "Welcome to",
    motto: "Your city, your story, your rules.",
    onlinePlayers: "Players online:",
    copyIp: "Copy Server IP",
    copied: "Copied!",
    joinDiscord: "Join Discord",
    shopTitle: "Our Shop",
    shopDescription: "In our shop, you will find unique items, vehicles, and much more to enrich your experience on the server. Every purchase supports the development of RedCity!",
    goToShop: "Go to Shop",
    rolesTitle: "Roles & Ranks",
    rolesDescription: "Want to be a part of the administration or join one of the factions? Check out the available roles and apply today to start your career in RedCity.",
    applyForRank: "Apply for a Rank",
    home: "Home",
    ranks: "Ranks",
    support: "Support",
  },
};

export default function HomePage() {
  const [ipCopied, setIpCopied] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [progress, setProgress] = useState(0);
  const [isScrolled, setIsScrolled] = useState(false);
  const [language, setLanguage] = useState("pl");
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);

  const t = translations[language];

  // Effect for loading screen
  useEffect(() => {
    const progressTimer = setInterval(() => {
      setProgress(prev => (prev >= 100 ? 100 : prev + 1));
    }, 30);

    const loadingTimer = setTimeout(() => {
      setIsLoading(false);
    }, 3500);

    return () => {
      clearInterval(progressTimer);
      clearTimeout(loadingTimer);
    };
  }, []);

  // Effect for navbar scroll
  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 10);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);
  
  // Effect to disable body scroll when mobile menu is open
  useEffect(() => {
    if (isMobileMenuOpen) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }
  }, [isMobileMenuOpen]);

  const serverIp = "127.0.0.1:30120"; // Placeholder for your server IP

  const handleCopyIp = () => {
    navigator.clipboard.writeText(serverIp);
    setIpCopied(true);
    setTimeout(() => {
      setIpCopied(false);
    }, 2000);
  };

  const toggleLanguage = () => {
    setLanguage(prev => prev === "pl" ? "en" : "pl");
  }

  const [currentFact, setCurrentFact] = useState(0);

  const funFacts = [
    "RedCity zostało założone w 2023 roku.",
    "Na serwerze znajduje się ponad 50 unikalnych pojazdów.",
    "Możesz podjąć pracę jako policjant, medyk, a nawet taksówkarz!",
    "Organizujemy cotygodniowe eventy z nagrodami.",
    "Nasza mapa jest stale rozwijana o nowe lokacje."
  ];

  const nextFact = () => {
    setCurrentFact((prev) => (prev + 1) % funFacts.length);
  };

  const prevFact = () => {
    setCurrentFact((prev) => (prev - 1 + funFacts.length) % funFacts.length);
  };

  if (isLoading) {
    return (
      <div
        className={`fixed inset-0 bg-background flex flex-col justify-center items-center z-50 transition-opacity duration-500 ${progress >= 100 ? "opacity-0" : "opacity-100"}`}>
        <div className="text-2xl font-bold text-primary mb-4">{t.loading}</div>
        <Progress value={progress} className="w-64" />
        <div className="absolute bottom-0 left-0 right-0 p-4 bg-card">
          <div className="flex items-center justify-center space-x-2 sm:space-x-4">
            <Button onClick={prevFact} variant="outline" size="icon">
              <FiChevronLeft className="h-6 w-6" />
            </Button>
            <Card className="w-full max-w-sm h-24 flex items-center justify-center mx-2 sm:mx-0">
              <CardContent className="p-4 text-center">
                <p className="text-base sm:text-lg text-card-foreground">{funFacts[currentFact]}</p>
              </CardContent>
            </Card>
            <Button onClick={nextFact} variant="outline" size="icon">
              <FiChevronRight className="h-6 w-6" />
            </Button>
          </div>
          <div className="flex justify-center space-x-2 mt-4">
            {funFacts.map((_, index) => (
              <div
                key={index}
                className={`w-2 h-2 rounded-full ${currentFact === index ? "bg-primary" : "bg-muted"}`}
              ></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-background text-foreground min-h-screen flex flex-col animate-fade-in">
      {/* Header */}
      <header className={`w-full p-4 fixed top-0 z-40 transition-all duration-300 ${isScrolled ? "bg-background/80 backdrop-blur-sm shadow-md border-b" : "bg-transparent"}`}>
        <div className="container mx-auto flex justify-between items-center">
          <div className="flex items-center space-x-2 sm:space-x-4">
            <Image src="/images/logo.png" alt="RedCity Logo" width={40} height={40} className="rounded-md" />
            <h1 className={`text-xl sm:text-2xl font-bold transition-colors duration-300 ${isScrolled ? "text-primary" : "text-white"}`}>
              RedCity RP
            </h1>
          </div>

          <div className="flex items-center space-x-2 md:space-x-4">
            <nav className="hidden md:flex space-x-6">
              <Button variant="link" asChild className={`font-semibold transition-colors duration-300 ${isScrolled ? "text-foreground hover:text-primary" : "text-white hover:text-gray-200"}`}><a href="#">{t.home}</a></Button>
              <Button variant="link" asChild className={`font-semibold transition-colors duration-300 ${isScrolled ? "text-foreground hover:text-primary" : "text-white hover:text-gray-200"}`}><a href="#">{t.ranks}</a></Button>
              <Button variant="link" asChild className={`font-semibold transition-colors duration-300 ${isScrolled ? "text-foreground hover:text-primary" : "text-white hover:text-gray-200"}`}><a href="#">{t.support}</a></Button>
            </nav>
            <div className="hidden md:flex items-center space-x-2">
                <Button asChild>
                  <Link href="/login">{t.playerPanel}</Link>
                </Button>
                <Button onClick={toggleLanguage} variant="outline" size="icon" className={`${isScrolled ? "" : "text-white border-white/50 hover:bg-white/20 hover:text-white"}`}>
                    <FiGlobe size={24}/>
                </Button>
            </div>
            <div className="md:hidden">
                <Button onClick={() => setIsMobileMenuOpen(true)} variant="ghost" size="icon" className={`${isScrolled ? "" : "text-white hover:bg-white/20 hover:text-white"}`}>
                    <FiMenu size={24} />
                </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Mobile Menu */}
      {isMobileMenuOpen && (
        <div className="fixed inset-0 bg-background z-50 flex flex-col items-center justify-center animate-fade-in">
            <Button onClick={() => setIsMobileMenuOpen(false)} variant="ghost" size="icon" className="absolute top-5 right-5">
                <FiX size={32} />
            </Button>
            <nav className="flex flex-col items-center space-y-8">
                <Button variant="link" asChild className="text-2xl font-semibold" onClick={() => setIsMobileMenuOpen(false)}><a href="#">{t.home}</a></Button>
                <Button variant="link" asChild className="text-2xl font-semibold" onClick={() => setIsMobileMenuOpen(false)}><a href="#">{t.ranks}</a></Button>
                <Button variant="link" asChild className="text-2xl font-semibold" onClick={() => setIsMobileMenuOpen(false)}><a href="#">{t.support}</a></Button>
                <Button asChild size="lg" className="mt-8" onClick={() => setIsMobileMenuOpen(false)}>
                    <Link href="/login">{t.playerPanel}</Link>
                </Button>
                <Button onClick={toggleLanguage} variant="outline">
                    <FiGlobe className="mr-2"/> {language === 'pl' ? 'English' : 'Polski'}
                </Button>
            </nav>
        </div>
      )}

      {/* Main Content */}
      <main className="flex-grow">
        {/* Hero Section */}
        <section className="relative min-h-screen flex items-center p-4 sm:p-8 overflow-hidden">
            <Image
                src="/images/background.png"
                alt="Tło miasta"
                fill
                className="object-cover grayscale"
            />
            <div className="absolute inset-0 bg-black/50 z-10"></div>
            <div className="relative z-20 container mx-auto">
                <Card className="max-w-xl text-center md:text-left">
                  <CardHeader>
                    <CardTitle className="text-5xl md:text-6xl font-extrabold">
                        {t.welcome} <span className="text-primary">RedCity</span>
                    </CardTitle>
                    <CardDescription className="text-xl pt-2">
                        {t.motto}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="flex justify-center md:justify-start items-center space-x-4 mb-8">
                        <div className="flex items-center space-x-2 bg-muted px-4 py-2 rounded-lg">
                        <FiUsers className="text-primary" />
                        <span className="font-semibold">{t.onlinePlayers}</span>
                        <span className="font-bold text-primary">? / ?</span>
                        </div>
                    </div>
                    <Button onClick={handleCopyIp} size="lg" className="w-full sm:w-64">
                      <FiCopy className="mr-2" />
                      {ipCopied ? t.copied : t.copyIp}
                    </Button>
                  </CardContent>
                </Card>
            </div>
            <div className="absolute bottom-0 right-16 z-20 hidden lg:block">
                <Image src="/images/businessman.png" alt="Postać" width={1000} height={1333} />
            </div>
        </section>

        {/* Shop Section */}
        <section className="py-20 bg-muted/40">
          <div className="container mx-auto px-8">
            <div className="grid md:grid-cols-2 gap-12 items-center">
                <div className="text-center md:text-left order-last md:order-first">
                  <h3 className="text-4xl font-bold mb-4">{t.shopTitle}</h3>
                  <p className="text-muted-foreground mb-6">
                    {t.shopDescription}
                  </p>
                  <Button asChild size="lg">
                    <a href="#">
                      <FiShoppingCart className="mr-2" />
                      {t.goToShop}
                    </a>
                  </Button>
                </div>
                <div className="flex justify-center">
                    <Image src="/images/your-way.webp" alt="Sklep RedCity" width={500} height={300} className="rounded-lg shadow-2xl"/>
                </div>
            </div>
          </div>
        </section>

        {/* Discord Section */}
        <section className="py-20 bg-background">
          <div className="container mx-auto px-8">
            <div className="grid md:grid-cols-2 gap-12 items-center">
                <div className="flex justify-center">
                    <Image src="/images/cars-clothes-every-taste.webp" alt="Samochody i ubrania" width={500} height={300} className="rounded-lg shadow-2xl"/>
                </div>
                <div className="text-center md:text-left">
                  <h3 className="text-4xl font-bold mb-4">{t.joinDiscord}</h3>
                  <p className="text-muted-foreground max-w-3xl mr-auto mb-8">
                    Dołącz do naszej społeczności na Discordzie, aby być na bieżąco z nowościami, wydarzeniami i poznać innych graczy!
                  </p>
                  <Button asChild size="lg" variant="secondary">
                    <a
                      href="https://discord.gg/your-invite-code"
                      target="_blank"
                      rel="noopener noreferrer"
                    >
                      <FaDiscord className="mr-2" />
                      {t.joinDiscord}
                    </a>
                  </Button>
                </div>
            </div>
          </div>
        </section>

        {/* Roles and Ranks Section */}
        <section className="py-20 bg-background">
          <div className="container mx-auto px-8">
            <div className="grid md:grid-cols-2 gap-12 items-center">
                <div className="text-center md:text-left order-last md:order-first">
                    <h3 className="text-4xl font-bold mb-4">{t.rolesTitle}</h3>
                    <p className="text-muted-foreground max-w-3xl mr-auto mb-8">
                      {t.rolesDescription}
                    </p>
                    <Button asChild size="lg" variant="secondary">
                      <a href="#">
                        <FiAward className="mr-2" />
                        {t.applyForRank}
                      </a>
                    </Button>
                </div>
                <div className="flex justify-center">
                    <Image src="/images/lots-of-entertainment.webp" alt="Dużo rozrywki" width={500} height={300} className="rounded-lg shadow-2xl"/>
                </div>
            </div>
          </div>
        </section>

        <Faq />

      </main>

      {/* Footer */}
      <footer className="w-full p-6 border-t bg-muted/40">
        <div className="container mx-auto flex justify-center items-center space-x-2">
          <Button variant="ghost" size="icon" asChild>
            <a href="#">
              <FaTiktok size={24} />
            </a>
          </Button>
          <Button variant="ghost" size="icon" asChild>
            <a href="#">
              <FaYoutube size={24} />
            </a>
          </Button>
        </div>
      </footer>
    </div>
  );
}
