"use client";
import { useState, useEffect } from "react";
import Image from "next/image";
import { useSession, signOut } from "next-auth/react";
import { useTheme } from "next-themes";
import { useRouter } from "next/navigation";
import { FiLogOut, FiShoppingCart, FiFileText, FiHelpCircle, FiLoader, FiCreditCard, FiX, FiAlertTriangle, FiRss, FiShield, FiUploadCloud, FiSettings, FiTool, FiUserX, FiTrash2, FiSearch, FiClock, FiMenu, FiUsers, FiBarChart2, FiGift, FiTag } from "react-icons/fi";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Slider } from "@/components/ui/slider";
import { Textarea } from "@/components/ui/textarea";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { Switch } from "@/components/ui/switch";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useAlertDialog } from "@/app/providers";
import { toast } from "sonner";
import Charts from "@/components/charts";

export default function PanelPage() {
  const { data: session, status } = useSession();
  const { setTheme, theme } = useTheme();
  const router = useRouter();
  const { showAlert, showConfirm } = useAlertDialog();
  const [activeTab, setActiveTab] = useState("portfel");
  const [supportMessage, setSupportMessage] = useState("");
  const [supportImage, setSupportImage] = useState<string | null>(null);
  const [severity, setSeverity] = useState("podstawowy");
  const vplnBalance = 100; // Placeholder for VPLN balance
  const [topUpAmount, setTopUpAmount] = useState(100);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedPayment, setSelectedPayment] = useState("blik");
  const [consentChecked, setConsentChecked] = useState(false);
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);

  const [announcements, setAnnouncements] = useState<any[]>([]);
  const [isLoadingAnnouncements, setIsLoadingAnnouncements] = useState(true);
  const [announcementTitle, setAnnouncementTitle] = useState("");
  const [announcementContent, setAnnouncementContent] = useState("");
  const [announcementImage, setAnnouncementImage] = useState<string | null>(null);
  const [announcementScheduledTime, setAnnouncementScheduledTime] = useState("");

  const [settings, setSettings] = useState<any>(null);
  const [editableSettings, setEditableSettings] = useState<any>(null);
  const [isLoadingSettings, setIsLoadingSettings] = useState(true);

  const [blacklist, setBlacklist] = useState<any[]>([]);
  const [isLoadingBlacklist, setIsLoadingBlacklist] = useState(true);
  const [newBlacklistId, setNewBlacklistId] = useState("");
  const [blacklistSearch, setBlacklistSearch] = useState("");
  const [currentUserPermissions, setCurrentUserPermissions] = useState<any>(null);

  const [admins, setAdmins] = useState<any[]>([]);
  const [isLoadingAdmins, setIsLoadingAdmins] = useState(true);
  const [newAdminId, setNewAdminId] = useState("");

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      const reader = new FileReader();
      reader.onloadend = () => {
        setAnnouncementImage(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const fetchAdmins = async () => {
    setIsLoadingAdmins(true);
    try {
      const response = await fetch('/api/admins');
      const data = await response.json();
      setAdmins(data);
    } catch (error) {
      console.error("Failed to fetch admins:", error);
    }
    setIsLoadingAdmins(false);
  };

  const handleAddAdmin = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!newAdminId.trim()) return;
    try {
      const response = await fetch('/api/admins', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ userId: newAdminId }),
      });
      if (response.ok) {
        setNewAdminId("");
        fetchAdmins();
      } else {
        const errorData = await response.json();
        showAlert("Błąd", errorData.error);
      }
    } catch (error) {
      showAlert("Błąd", "Wystąpił błąd sieci.");
    }
  };

  const handleRemoveAdmin = async (userId: string) => {
    showConfirm("Potwierdzenie", "Czy na pewno chcesz usunąć tego administratora?", async () => {
      try {
        const response = await fetch('/api/admins', {
          method: 'DELETE',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ userId }),
        });
        if (response.ok) {
          fetchAdmins();
        } else {
          showAlert("Błąd", "Błąd podczas usuwania.");
        }
      } catch (error) {
        showAlert("Błąd", "Wystąpił błąd sieci.");
      }
    });
  };

  const fetchSettings = async () => {
    try {
      const response = await fetch('/api/settings');
      const data = await response.json();
      setSettings(data);
      setEditableSettings(data);
    } catch (error) {
      console.error("Failed to fetch settings:", error);
    }
    setIsLoadingSettings(false);
  };

  const handleSettingsInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    const [parent, child] = name.split('.');

    if (child) {
      setEditableSettings((prev: any) => ({
        ...prev,
        [parent]: {
          ...prev[parent],
          [child]: value,
        },
      }));
    } else {
      setEditableSettings((prev: any) => ({
        ...prev,
        [name]: value,
      }));
    }
  };

  const handleSettingsSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    try {
      const response = await fetch('/api/settings', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(editableSettings),
      });

      if (response.ok) {
        showAlert("Sukces", "Ustawienia zostały zapisane!");
        const data = await response.json();
        setSettings(data.settings);
      } else {
        showAlert("Błąd", "Wystąpił błąd podczas zapisywania ustawień.");
      }
    } catch (error) {
      console.error("Failed to save settings:", error);
      showAlert("Błąd", "Wystąpił błąd podczas zapisywania ustawień.");
    }
  };

  const fetchBlacklist = async () => {
    setIsLoadingBlacklist(true);
    try {
      const response = await fetch('/api/blacklist');
      const data = await response.json();
      setBlacklist(data);
    } catch (error) {
      console.error("Failed to fetch blacklist:", error);
    }
    setIsLoadingBlacklist(false);
  };

  const handleAddToBlacklist = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!newBlacklistId.trim()) return;
    try {
      const response = await fetch('/api/blacklist', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ userId: newBlacklistId }),
      });
      if (response.ok) {
        setNewBlacklistId("");
        fetchBlacklist();
      } else {
        const errorData = await response.json();
        showAlert("Błąd", errorData.error);
      }
    } catch (error) {
      showAlert("Błąd", "Wystąpił błąd sieci.");
    }
  };

  const handleRemoveFromBlacklist = async (userId: string) => {
    showConfirm("Potwierdzenie", "Czy na pewno chcesz usunąć tego użytkownika z czarnej listy?", async () => {
      try {
        const response = await fetch('/api/blacklist', {
          method: 'DELETE',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ userId }),
        });
        if (response.ok) {
          fetchBlacklist();
        } else {
          showAlert("Błąd", "Błąd podczas usuwania.");
        }
      } catch (error) {
        showAlert("Błąd", "Wystąpił błąd sieci.");
      }
    });
  };

  const handleBlacklistPermissionChange = async (userId: string, permission: string, value: boolean) => {
    try {
      await fetch('/api/blacklist', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ userId, permissions: { [permission]: value } }),
      });
      // Optimistically update UI, or refetch for consistency
      fetchBlacklist();
    } catch (error) {
      showAlert("Błąd", "Błąd podczas aktualizacji uprawnień.");
    }
  };

  const handleSupportSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!supportMessage.trim()) {
      showAlert("Błąd", "Wiadomość nie może być pusta.");
      return;
    }

    try {
      const response = await fetch('/api/support', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ message: supportMessage, user: session?.user, severity, image: supportImage }),
      });

      if (response.ok) {
        showAlert("Sukces", "Twoje zgłoszenie zostało wysłane. Jeśli sprawa jest prywatna, stwórz ticket i czekaj na odpowiedź administracji.");
        setSupportMessage("");
        setSeverity("podstawowy");
      } else {
        const errorData = await response.json();
        showAlert("Błąd", errorData.error);
      }
    } catch (error) {
      console.error("Błąd podczas wysyłania zgłoszenia:", error);
      showAlert("Błąd", "Wystąpił błąd podczas wysyłania zgłoszenia. Spróbuj ponownie później.");
    }
  };

  const fetchAnnouncements = async () => {
    setIsLoadingAnnouncements(true);
    try {
      const url = (session?.user as any)?.isAdmin ? '/api/announcements?include_all=true' : '/api/announcements';
      const response = await fetch(url);
      const data = await response.json();
      setAnnouncements(data);
    } catch (error) {
      console.error("Failed to fetch announcements:", error);
      showAlert("Błąd", "Nie udało się wczytać ogłoszeń.");
    }
    setIsLoadingAnnouncements(false);
  };

  const handleAnnouncementSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!announcementTitle.trim() || !announcementContent.trim()) {
      toast.error("Tytuł i treść ogłoszenia nie mogą być puste.");
      return;
    }

    try {
      const response = await fetch('/api/announcements', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ 
          title: announcementTitle, 
          content: announcementContent, 
          image: announcementImage,
          scheduledTime: announcementScheduledTime || null 
        }),
      });

      if (response.ok) {
        toast.success("Ogłoszenie zostało dodane!");
        setAnnouncementTitle("");
        setAnnouncementContent("");
        setAnnouncementImage(null);
        setAnnouncementScheduledTime("");
        fetchAnnouncements();
      } else {
        const errorData = await response.json();
        toast.error(errorData.error);
      }
    } catch (error) {
      console.error("Błąd podczas dodawania ogłoszenia:", error);
      toast.error("Wystąpił błąd podczas dodawania ogłoszenia.");
    }
  };

  const handleDeleteAnnouncement = async (id: number) => {
    showConfirm("Potwierdzenie", "Czy na pewno chcesz usunąć to ogłoszenie?", async () => {
      try {
        const response = await fetch(`/api/announcements?id=${id}`, {
          method: 'DELETE',
        });
        if (response.ok) {
          toast.success("Ogłoszenie zostało usunięte.");
          fetchAnnouncements();
        } else {
          const errorData = await response.json();
          toast.error(errorData.error || "Błąd podczas usuwania ogłoszenia.");
        }
      } catch (error) {
        toast.error("Wystąpił błąd sieci podczas usuwania ogłoszenia.");
      }
    });
  };

  useEffect(() => {
    if (status === "authenticated" && activeTab === 'ogloszenia') {
      fetchAnnouncements();
    }
    if (status === "authenticated" && activeTab === 'administratorzy') {
      fetchAdmins();
    }
  }, [activeTab, status]);

  useEffect(() => {
    fetchSettings();
    fetchBlacklist();
  }, []);

  useEffect(() => {
    if (status === "authenticated" && session?.user && blacklist.length > 0) {
      const userOnBlacklist = blacklist.find(u => u.id === (session.user as any).id);
      if (userOnBlacklist) {
        setCurrentUserPermissions(userOnBlacklist.permissions);

        // Update user info on the blacklist if it's missing
        if (!userOnBlacklist.userInfo) {
          const userInfo = {
            name: session.user.name,
            email: session.user.email,
            image: session.user.image,
          };
          fetch('/api/blacklist', {
            method: 'PUT',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ userId: (session.user as any).id, userInfo }),
          });
        }
      }
    }
  }, [status, session, blacklist]);

  useEffect(() => {
    if (status === "unauthenticated") {
      router.push("/login");
    }
  }, [status, router]);

  if (status === "loading" || isLoadingSettings) {
    return (
      <div className="flex justify-center items-center h-screen bg-gray-50">
        <FiLoader className="animate-spin text-red-600" size={48} />
      </div>
    );
  }

  if (settings?.maintenanceMode && !(session?.user as any)?.isAdmin) {
    return (
      <div className="flex flex-col justify-center items-center h-screen bg-gray-50 text-gray-800">
        <FiTool size={64} className="text-red-600 mb-4" />
        <h1 className="text-4xl font-bold mb-2">Trwają prace konserwacyjne</h1>
        <p className="text-lg">Przepraszamy za utrudnienia. Wracamy wkrótce!</p>
      </div>
    );
  }

  if (status === "authenticated") {
    return (
      <>
        <div className="flex h-screen bg-background text-foreground animate-fade-in">
          {/* Sidebar */}
          <div
            className={`fixed inset-y-0 left-0 z-50 w-64 bg-card border-r flex flex-col p-4 transform ${
              isSidebarOpen ? "translate-x-0" : "-translate-x-full"
            } transition-transform duration-300 ease-in-out md:relative md:translate-x-0 md:inset-y-auto md:left-auto`}
          >
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center space-x-4">
                <Image src="/images/logo.png" alt="RedCity Logo" width={40} height={40} className="rounded-lg" />
                <h1 className="text-xl font-bold text-primary">
                  {settings?.siteTitle || 'RedCity RP'}
                </h1>
              </div>
              <Button variant="ghost" size="icon" className="md:hidden" onClick={() => setIsSidebarOpen(false)}>
                <FiX className="h-6 w-6" />
              </Button>
            </div>

            <nav className="flex-grow space-y-1">
              <Button variant={activeTab === "sklep" ? "secondary" : "ghost"} onClick={() => { setActiveTab("sklep"); setIsSidebarOpen(false); }} className="w-full justify-start">
                <FiShoppingCart className="mr-2 h-4 w-4" />
                Sklep
              </Button>
              <Button variant={activeTab === "portfel" ? "secondary" : "ghost"} onClick={() => { setActiveTab("portfel"); setIsSidebarOpen(false); }} className="w-full justify-start">
                <FiCreditCard className="mr-2 h-4 w-4" />
                Doładuj portfel
              </Button>
              <Button variant={activeTab === "podania" ? "secondary" : "ghost"} onClick={() => { setActiveTab("podania"); setIsSidebarOpen(false); }} className="w-full justify-start">
                <FiFileText className="mr-2 h-4 w-4" />
                Podania
              </Button>
              <Button variant={activeTab === "ogloszenia" ? "secondary" : "ghost"} onClick={() => { setActiveTab("ogloszenia"); setIsSidebarOpen(false); }} className="w-full justify-start">
                <FiRss className="mr-2 h-4 w-4" />
                Ogłoszenia
              </Button>
              <Button variant={activeTab === "support" ? "secondary" : "ghost"} onClick={() => { setActiveTab("support"); setIsSidebarOpen(false); }} className="w-full justify-start">
                <FiHelpCircle className="mr-2 h-4 w-4" />
                Support
              </Button>
              {(session?.user as any)?.isAdmin && (
                <div className="pt-4 mt-4 border-t">
                  <h3 className="px-4 text-xs font-semibold text-muted-foreground uppercase tracking-wider mb-2">Admin</h3>
                  <div className="space-y-1">
                    <Button variant={activeTab === "administracja" ? "secondary" : "ghost"} onClick={() => { setActiveTab("administracja"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiShield className="mr-2 h-4 w-4" />
                      Administracja
                    </Button>
                    <Button variant={activeTab === "ustawienia" ? "secondary" : "ghost"} onClick={() => { setActiveTab("ustawienia"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiSettings className="mr-2 h-4 w-4" />
                      Ustawienia Strony
                    </Button>
                    <Button variant={activeTab === "czarna-lista" ? "secondary" : "ghost"} onClick={() => { setActiveTab("czarna-lista"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiUserX className="mr-2 h-4 w-4" />
                      Czarna Lista
                    </Button>
                    <Button variant={activeTab === "administratorzy" ? "secondary" : "ghost"} onClick={() => { setActiveTab("administratorzy"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiUsers className="mr-2 h-4 w-4" />
                      Administratorzy
                    </Button>
                    <Button variant={activeTab === "statystyki" ? "secondary" : "ghost"} onClick={() => { setActiveTab("statystyki"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiBarChart2 className="mr-2 h-4 w-4" />
                      Statystyki
                    </Button>
                    <Button variant={activeTab === "kody-rabatowe" ? "secondary" : "ghost"} onClick={() => { setActiveTab("kody-rabatowe"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiTag className="mr-2 h-4 w-4" />
                      Kody rabatowe
                    </Button>
                    <Button variant={activeTab === "vouchery" ? "secondary" : "ghost"} onClick={() => { setActiveTab("vouchery"); setIsSidebarOpen(false); }} className="w-full justify-start">
                      <FiGift className="mr-2 h-4 w-4" />
                      Vouchery
                    </Button>
                  </div>
                </div>
              )}
            </nav>

            {/* User Info */}
            <div className="mt-auto">
              <div className="p-3 bg-muted rounded-lg">
                <div className="flex items-center space-x-3">
                  <Avatar>
                    <AvatarImage src={session.user?.image || undefined} />
                    <AvatarFallback>{session.user?.name?.charAt(0)}</AvatarFallback>
                  </Avatar>
                  <div>
                    <p className="font-semibold text-sm text-foreground">{session.user?.name}</p>
                    <p className="font-bold text-sm text-primary">{vplnBalance} VPLN</p>
                  </div>
                </div>
                <Button
                  onClick={() => signOut({ callbackUrl: "/" })}
                  variant="ghost"
                  size="sm"
                  className="w-full justify-center mt-2"
                >
                  <FiLogOut className="mr-2 h-4 w-4" />
                  Wyloguj
                </Button>
              </div>
            </div>
          </div>

          {/* Main Content */}
          <main className="flex-1 p-4 md:p-10 overflow-y-auto">
            <div className="flex items-center justify-between md:hidden mb-4">
              <Button variant="ghost" size="icon" onClick={() => setIsSidebarOpen(true)}>
                <FiMenu className="h-6 w-6" />
              </Button>
              <div className="flex items-center space-x-2">
                <Image src="/images/logo.png" alt="RedCity Logo" width={32} height={32} className="rounded-lg" />
                <h1 className="text-lg font-bold text-primary">
                  {settings?.siteTitle || 'RedCity RP'}
                </h1>
              </div>
            </div>
            <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
              <TabsContent value="sklep">
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
                  {
                    [
                      { name: 'Sportowe Auto', price: '250 VPLN', image: '/images/cars-clothes-every-taste.webp' },
                      { name: 'Elegancki Garnitur', price: '50 VPLN', image: '/images/your-way.webp' },
                      { name: 'Apartament w Centrum', price: '500 VPLN', image: '/images/lots-of-entertainment.webp' },
                      { name: 'Paczka Gotówki', price: '100 VPLN', image: '/images/background.png' },
                    ].map((item, index) => (
                      <Card key={index}>
                        <CardHeader className="p-0">
                          <Image src={item.image} alt={item.name} width={400} height={300} className="rounded-t-lg object-cover w-full h-48" />
                        </CardHeader>
                        <CardContent className="p-4">
                          <CardTitle className="text-lg">{item.name}</CardTitle>
                          <p className="text-primary font-bold">{item.price}</p>
                        </CardContent>
                        <CardFooter className="p-4 pt-0">
                          <Button className="w-full">Kup</Button>
                        </CardFooter>
                      </Card>
                    ))
                  }
                </div>
              </TabsContent>
              <TabsContent value="portfel">
                <Card className="max-w-2xl mx-auto">
                  <CardHeader>
                    <CardTitle>Doładuj swój portfel</CardTitle>
                    <CardDescription>Wybierz kwotę doładowania VPLN, którą chcesz dodać do swojego konta.</CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="mb-6">
                      <Label htmlFor="vpln-slider" className="text-lg">Wybierz kwotę doładowania:</Label>
                      <div className="flex flex-col sm:flex-row items-center space-y-4 sm:space-y-0 sm:space-x-4 pt-4">
                        <Slider
                          id="vpln-slider"
                          min={1}
                          max={400}
                          step={1}
                          value={[topUpAmount]}
                          onValueChange={(value) => setTopUpAmount(value[0])}
                          className="flex-1"
                        />
                        <span className="font-bold text-xl text-primary w-32 text-center">{topUpAmount} VPLN</span>
                      </div>
                    </div>
                    <div className="flex items-start space-x-2 mb-4">
                      <input type="checkbox" id="consent" checked={consentChecked} onChange={(e) => setConsentChecked(e.target.checked)} />
                      <label htmlFor="consent" className="text-sm text-muted-foreground">
                        Wyrażam zgodę na natychmiastowe dostarczenie usługi i rozumiem, że jeśli płatność zostanie zrealizowana to nie przysługuje mi prawo do odstąpienia od umowy zawartej elektronicznie.
                      </label>
                    </div>
                    <div className="text-center">
                      <p className="text-muted-foreground mb-4">Cena: <span className="font-bold">{topUpAmount} PLN</span></p>
                      <Button onClick={() => setIsModalOpen(true)} size="lg" disabled={!consentChecked}>
                        Przejdź do płatności
                      </Button>
                      <p className="text-xs text-muted-foreground mt-2">Cena zawiera już 23% podatku VAT.</p>
                    </div>
                  </CardContent>
                </Card>
                <Card className="max-w-2xl mx-auto mt-6">
                  <CardHeader>
                    <CardTitle>Kody rabatowe i Vouchery</CardTitle>
                    <CardDescription>Masz kod rabatowy lub voucher? Wpisz go tutaj.</CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="flex items-center space-x-2">
                      <Input type="text" placeholder="Wpisz kod..." />
                      <Button>Aktywuj</Button>
                    </div>
                  </CardContent>
                </Card>
              </TabsContent>
              <TabsContent value="podania">
                {currentUserPermissions && !currentUserPermissions.canUseApplications ? (
                  <Card className="max-w-2xl mx-auto text-center">
                    <CardHeader>
                      <div className="mx-auto bg-destructive/10 p-4 rounded-full w-fit">
                        <FiUserX size={48} className="text-destructive" />
                      </div>
                      <CardTitle className="pt-4">Dostęp Zablokowany</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <p className="text-muted-foreground">Nie masz uprawnień do korzystania z tej sekcji. Zostałeś zablokowany przez administratora.</p>
                    </CardContent>
                  </Card>
                ) : (
                  <div>Zawartość podań</div>
                )}
              </TabsContent>
              <TabsContent value="ogloszenia">
                <div className="max-w-4xl mx-auto">
                  <h2 className="text-3xl font-bold mb-6">Ogłoszenia</h2>
                  {isLoadingAnnouncements ? (
                    <div className="flex justify-center items-center p-10">
                      <FiLoader className="animate-spin text-primary" size={32} />
                    </div>
                  ) : announcements.length > 0 ? (
                    <div className="space-y-6">
                      {announcements.map((ann) => (
                        <Card key={ann.id} className="animate-fade-in relative flex flex-col md:flex-row">
                          {(session?.user as any)?.isAdmin && (
                            <div className="absolute top-4 right-4 flex items-center space-x-2">
                              <Button onClick={() => handleDeleteAnnouncement(ann.id)} variant="destructive" size="icon">
                                <FiTrash2 className="h-4 w-4" />
                              </Button>
                            </div>
                          )}
                          {ann.image && (
                            <div className="md:w-1/3">
                              <Image src={ann.image} alt={ann.title} width={800} height={400} className="rounded-t-lg md:rounded-l-lg md:rounded-t-none object-cover w-full h-full" />
                            </div>
                          )}
                          <div className="flex flex-col justify-between md:w-2/3">
                            <div>
                              <CardHeader>
                                <CardTitle>{ann.title}</CardTitle>
                                {ann.scheduledTime && new Date(ann.scheduledTime) > new Date() && (
                                  <div className="flex items-center text-sm text-amber-500">
                                    <FiClock className="mr-2" />
                                    Zaplanowano na: {new Date(ann.scheduledTime).toLocaleString('pl-PL')}
                                  </div>
                                )}
                              </CardHeader>
                              <CardContent>
                                <p className="text-muted-foreground whitespace-pre-wrap">{ann.content}</p>
                              </CardContent>
                            </div>
                            <CardFooter className="text-xs text-muted-foreground justify-end space-x-4 mt-auto">
                              <span>Autor: <span className="font-semibold">{ann.author}</span></span>
                              <span>{new Date(ann.createdAt).toLocaleString('pl-PL')}</span>
                            </CardFooter>
                          </div>
                        </Card>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center text-muted-foreground py-10">Brak ogłoszeń.</div>
                  )}
                </div>
              </TabsContent>
              <TabsContent value="support">
                {currentUserPermissions && !currentUserPermissions.canUseSupport ? (
                  <Card className="max-w-2xl mx-auto text-center">
                    <CardHeader>
                      <div className="mx-auto bg-destructive/10 p-4 rounded-full w-fit">
                        <FiUserX size={48} className="text-destructive" />
                      </div>
                      <CardTitle className="pt-4">Dostęp Zablokowany</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <p className="text-muted-foreground">Nie masz uprawnień do korzystania z tej sekcji. Zostałeś zablokowany przez administratora.</p>
                    </CardContent>
                  </Card>
                ) : (
                  <Card className="max-w-2xl mx-auto">
                    <CardHeader>
                      <CardTitle>Support</CardTitle>
                      <CardDescription>Potrzebujesz pomocy? Opisz swój problem, a my postaramy się jak najszybciej odpowiedzieć.</CardDescription>
                    </CardHeader>
                    <CardContent>
                      <form onSubmit={handleSupportSubmit}>
                        <div className="grid w-full gap-4">
                          <div className="flex flex-col space-y-1.5">
                            <Label htmlFor="support-message" className="text-lg">Twoja wiadomość:</Label>
                            <Textarea id="support-message" rows={6} value={supportMessage} onChange={(e) => setSupportMessage(e.target.value)} placeholder="Opisz swój problem..." />
                          </div>
                          <div className="flex flex-col space-y-1.5">
                            <Label htmlFor="support-image" className="text-lg">Dodaj zrzut ekranu (opcjonalnie):</Label>
                            <Input id="support-image" type="file" accept="image/*" onChange={(e) => {
                              if (e.target.files && e.target.files[0]) {
                                const file = e.target.files[0];
                                const reader = new FileReader();
                                reader.onloadend = () => {
                                  setSupportImage(reader.result as string);
                                };
                                reader.readAsDataURL(file);
                              }
                            }} />
                            {supportImage && (
                              <div className="mt-4">
                                <p className="text-sm font-semibold mb-2">Podgląd:</p>
                                <Image src={supportImage} alt="Podgląd obrazka" width={500} height={300} className="rounded-lg object-cover w-full" />
                              </div>
                            )}
                          </div>
                          <div className="flex flex-col space-y-1.5">
                            <Label className="text-lg">Waga zgłoszenia:</Label>
                            <RadioGroup value={severity} onValueChange={setSeverity} className="p-2 bg-muted rounded-lg flex flex-wrap gap-2">
                              <Label className="flex-1 flex items-center justify-center space-x-2 cursor-pointer p-2 rounded-md hover:bg-background transition-colors has-[:checked]:bg-background">
                                <RadioGroupItem value="podstawowy" id="r1" />
                                <span className="font-semibold">Podstawowy</span>
                              </Label>
                              <Label className="flex-1 flex items-center justify-center space-x-2 cursor-pointer p-2 rounded-md hover:bg-background transition-colors has-[:checked]:bg-background">
                                <RadioGroupItem value="średni" id="r2" />
                                <span className="font-semibold">Średni</span>
                              </Label>
                              <Label className="flex-1 flex items-center justify-center space-x-2 cursor-pointer p-2 rounded-md hover:bg-background transition-colors has-[:checked]:bg-background">
                                <RadioGroupItem value="poważny" id="r3" />
                                <span className="font-semibold">Poważny</span>
                              </Label>
                            </RadioGroup>
                          </div>
                          {(severity === 'średni' || severity === 'poważny') && (
                            <Alert variant="destructive">
                              <FiAlertTriangle className="h-4 w-4" />
                              <AlertTitle>Uwaga!</AlertTitle>
                              <AlertDescription>
                                Bezpodstawne wysyłanie zgłoszenia o tej wadze grozi banem.
                              </AlertDescription>
                            </Alert>
                          )}
                          <div className="text-center">
                            <Button type="submit" size="lg">
                              Wyślij zgłoszenie
                            </Button>
                          </div>
                        </div>
                      </form>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>
              {(session?.user as any)?.isAdmin && (
                <>
                  <TabsContent value="administracja">
                    <Card className="max-w-2xl mx-auto">
                      <CardHeader>
                        <CardTitle>Dodaj Nowe Ogłoszenie</CardTitle>
                        <CardDescription>Stwórz i opublikuj nowe ogłoszenie dla graczy.</CardDescription>
                      </CardHeader>
                      <CardContent>
                        <form onSubmit={handleAnnouncementSubmit} className="space-y-6">
                          <div className="space-y-2">
                            <Label htmlFor="ann-title" className="text-lg">Tytuł:</Label>
                            <Input
                              id="ann-title"
                              type="text"
                              value={announcementTitle}
                              onChange={(e) => setAnnouncementTitle(e.target.value)}
                              placeholder="Tytuł ogłoszenia..."
                            />
                          </div>
                          <div className="space-y-2">
                            <Label htmlFor="ann-content" className="text-lg">Treść:</Label>
                            <Textarea
                              id="ann-content"
                              rows={8}
                              value={announcementContent}
                              onChange={(e) => setAnnouncementContent(e.target.value)}
                              placeholder="Treść ogłoszenia..."
                            />
                          </div>
                          <div className="space-y-2">
                            <Label htmlFor="ann-scheduled-time" className="text-lg">Zaplanuj publikację (opcjonalnie):</Label>
                            <Input
                              id="ann-scheduled-time"
                              type="datetime-local"
                              value={announcementScheduledTime}
                              onChange={(e) => setAnnouncementScheduledTime(e.target.value)}
                            />
                          </div>
                          <div className="space-y-2">
                            <Label className="text-lg">Obrazek (opcjonalnie):</Label>
                            <Label htmlFor="ann-image" className="w-full flex justify-center items-center space-x-2 p-4 bg-muted rounded-lg cursor-pointer hover:bg-muted/80 transition-colors border-2 border-dashed border-muted-foreground/50">
                              <FiUploadCloud size={24} />
                              <span>Wybierz plik</span>
                            </Label>
                            <Input
                              id="ann-image"
                              type="file"
                              accept="image/*"
                              className="hidden"
                              onChange={handleImageChange}
                            />
                            {announcementImage && (
                              <div className="mt-4">
                                <p className="text-sm font-semibold mb-2">Podgląd:</p>
                                <Image src={announcementImage} alt="Podgląd obrazka" width={500} height={300} className="rounded-lg object-cover w-full" />
                              </div>
                            )}
                          </div>
                          <div className="text-center">
                            <Button type="submit" size="lg">
                              Opublikuj Ogłoszenie
                            </Button>
                          </div>
                        </form>
                      </CardContent>
                    </Card>
                  </TabsContent>
                  <TabsContent value="ustawienia">
                    <Card className="max-w-2xl mx-auto">
                      <CardHeader>
                        <CardTitle>Ustawienia Strony</CardTitle>
                        <CardDescription>Zarządzaj globalnymi ustawieniami witryny.</CardDescription>
                      </CardHeader>
                      <CardContent>
                        {editableSettings ? (
                          <form onSubmit={handleSettingsSubmit} className="space-y-8">
                            {/* Maintenance Mode */}
                            <div className="p-6 border rounded-lg bg-muted/50">
                              <div className="flex items-center justify-between">
                                <div>
                                  <Label htmlFor="maintenance-mode" className="text-lg font-semibold">Tryb Konserwacyjny</Label>
                                  <p className="text-sm text-muted-foreground mt-1">Gdy włączone, użytkownicy zobaczą stronę prac technicznych.</p>
                                </div>
                                <Switch
                                  id="maintenance-mode"
                                  checked={editableSettings.maintenanceMode}
                                  onCheckedChange={(checked) => setEditableSettings((prev: any) => ({...prev, maintenanceMode: checked}))}
                                />
                              </div>
                            </div>

                            {/* Theme Switcher */}
                            <div className="p-6 border rounded-lg bg-muted/50">
                              <div className="flex items-center justify-between">
                                <div>
                                  <Label htmlFor="theme-switcher" className="text-lg font-semibold">Motyw</Label>
                                  <p className="text-sm text-muted-foreground mt-1">Wybierz motyw dla strony.</p>
                                </div>
                                <Select onValueChange={setTheme} defaultValue={theme}>
                                  <SelectTrigger className="w-[180px]">
                                    <SelectValue placeholder="Wybierz motyw" />
                                  </SelectTrigger>
                                  <SelectContent>
                                    <SelectItem value="light">Jasny</SelectItem>
                                    <SelectItem value="dark">Ciemny</SelectItem>
                                    <SelectItem value="system">Systemowy</SelectItem>
                                  </SelectContent>
                                </Select>
                              </div>
                            </div>

                            {/* Site Title */}
                            <div className="space-y-2">
                              <Label htmlFor="siteTitle" className="text-lg">Tytuł Strony</Label>
                              <Input
                                id="siteTitle"
                                name="siteTitle"
                                type="text"
                                value={editableSettings.siteTitle}
                                onChange={handleSettingsInputChange}
                              />
                            </div>

                            {/* Social Links */}
                            <div className="space-y-4">
                              <h3 className="text-lg font-semibold">Linki Społecznościowe</h3>
                              <div className="space-y-3">
                                {Object.keys(editableSettings.socialLinks).map((key) => (
                                  <div key={key} className="space-y-1">
                                    <Label htmlFor={`social-${key}`} className="capitalize text-sm font-medium">{key}</Label>
                                    <Input
                                      id={`social-${key}`}
                                      name={`socialLinks.${key}`}
                                      type="text"
                                      value={editableSettings.socialLinks[key]}
                                      onChange={handleSettingsInputChange}
                                      placeholder={`https://...`}
                                    />
                                  </div>
                                ))}
                              </div>
                            </div>

                            <div className="text-center mt-8">
                              <Button type="submit" size="lg">
                                Zapisz Ustawienia
                              </Button>
                            </div>
                          </form>
                        ) : (
                          <div className="flex justify-center items-center p-10">
                            <FiLoader className="animate-spin text-primary" size={32} />
                          </div>
                        )}
                      </CardContent>
                    </Card>
                  </TabsContent>
                  <TabsContent value="czarna-lista">
                    <Card className="max-w-4xl mx-auto">
                      <CardHeader>
                        <CardTitle>Czarna Lista</CardTitle>
                        <CardDescription>Zarządzaj użytkownikami na czarnej liście i ich uprawnieniami.</CardDescription>
                      </CardHeader>
                      <CardContent className="space-y-6">
                        {/* Add User Form */}
                        <form onSubmit={handleAddToBlacklist} className="flex items-center space-x-4">
                          <Input
                            type="text"
                            value={newBlacklistId}
                            onChange={(e) => setNewBlacklistId(e.target.value)}
                            placeholder="Wpisz ID Discord użytkownika..."
                          />
                          <Button type="submit">Dodaj</Button>
                        </form>

                        {/* Search Bar */}
                        <div className="relative">
                          <FiSearch className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                          <Input
                            type="text"
                            value={blacklistSearch}
                            onChange={(e) => setBlacklistSearch(e.target.value)}
                            placeholder="Szukaj po nicku lub ID..."
                            className="pl-10"
                          />
                        </div>

                        {/* Blacklist Table */}
                        <div className="border rounded-lg">
                          <Table>
                            <TableHeader>
                              <TableRow>
                                <TableHead>Użytkownik</TableHead>
                                <TableHead className="hidden md:table-cell">Blokada Supportu</TableHead>
                                <TableHead className="hidden md:table-cell">Blokada Podań</TableHead>
                                <TableHead className="text-right">Akcje</TableHead>
                              </TableRow>
                            </TableHeader>
                            <TableBody>
                              {isLoadingBlacklist ? (
                                <TableRow>
                                  <TableCell colSpan={4} className="h-24 text-center">
                                    <FiLoader className="animate-spin text-primary mx-auto" size={24} />
                                  </TableCell>
                                </TableRow>
                              ) : (blacklist
                                  .filter(user => 
                                    user.id.includes(blacklistSearch) || 
                                    (user.userInfo?.name && user.userInfo.name.toLowerCase().includes(blacklistSearch.toLowerCase()))
                                  )
                                  .map(user => (
                                    <TableRow key={user.id}>
                                      <TableCell>
                                        <div className="flex items-center space-x-4">
                                          <Avatar>
                                            <AvatarImage src={user.userInfo?.image || undefined} />
                                            <AvatarFallback>{user.userInfo?.name?.charAt(0) || '?'}</AvatarFallback>
                                          </Avatar>
                                          <div>
                                            <p className="font-bold">{user.userInfo?.name || "Brak danych"}</p>
                                            <p className="text-sm text-muted-foreground">ID: {user.id}</p>
                                          </div>
                                        </div>
                                      </TableCell>
                                      <TableCell className="hidden md:table-cell">
                                        <Switch
                                          checked={!user.permissions.canUseSupport}
                                          onCheckedChange={(checked) => handleBlacklistPermissionChange(user.id, 'canUseSupport', !checked)}
                                        />
                                      </TableCell>
                                      <TableCell className="hidden md:table-cell">
                                        <Switch
                                          checked={!user.permissions.canUseApplications}
                                          onCheckedChange={(checked) => handleBlacklistPermissionChange(user.id, 'canUseApplications', !checked)}
                                        />
                                      </TableCell>
                                      <TableCell className="text-right">
                                        <Button onClick={() => handleRemoveFromBlacklist(user.id)} variant="ghost" size="icon">
                                          <FiTrash2 className="h-4 w-4" />
                                        </Button>
                                      </TableCell>
                                    </TableRow>
                                  )))}
                            </TableBody>
                          </Table>
                        </div>
                      </CardContent>
                    </Card>
                  </TabsContent>
                  <TabsContent value="administratorzy">
                    <Card className="max-w-4xl mx-auto">
                      <CardHeader>
                        <CardTitle>Administratorzy</CardTitle>
                        <CardDescription>Zarządzaj administratorami i ich uprawnieniami.</CardDescription>
                      </CardHeader>
                      <CardContent className="space-y-6">
                        {/* Add Admin Form */}
                        <form onSubmit={handleAddAdmin} className="flex items-center space-x-4">
                          <Input
                            type="text"
                            value={newAdminId}
                            onChange={(e) => setNewAdminId(e.target.value)}
                            placeholder="Wpisz ID Discord użytkownika..."
                          />
                          <Button type="submit">Dodaj Administratora</Button>
                        </form>

                        {/* Admins Table */}
                        <div className="border rounded-lg">
                          <Table>
                            <TableHeader>
                              <TableRow>
                                <TableHead>Użytkownik</TableHead>
                                <TableHead className="text-right">Akcje</TableHead>
                              </TableRow>
                            </TableHeader>
                            <TableBody>
                              {isLoadingAdmins ? (
                                <TableRow>
                                  <TableCell colSpan={2} className="h-24 text-center">
                                    <FiLoader className="animate-spin text-primary mx-auto" size={24} />
                                  </TableCell>
                                </TableRow>
                              ) : (admins.map(admin => (
                                    <TableRow key={admin.id}>
                                      <TableCell>
                                        <div className="flex items-center space-x-4">
                                          <Avatar>
                                            <AvatarImage src={admin.userInfo?.image || undefined} />
                                            <AvatarFallback>{admin.userInfo?.name?.charAt(0) || '?'}</AvatarFallback>
                                          </Avatar>
                                          <div>
                                            <p className="font-bold">{admin.userInfo?.name || "Brak danych"}</p>
                                            <p className="text-sm text-muted-foreground">ID: {admin.id}</p>
                                          </div>
                                        </div>
                                      </TableCell>
                                      <TableCell className="text-right">
                                        <Button onClick={() => handleRemoveAdmin(admin.id)} variant="ghost" size="icon">
                                          <FiTrash2 className="h-4 w-4" />
                                        </Button>
                                      </TableCell>
                                    </TableRow>
                                  )))}
                            </TableBody>
                          </Table>
                        </div>
                      </CardContent>
                    </Card>
                  </TabsContent>
                  <TabsContent value="statystyki">
                    <Charts />
                  </TabsContent>
                  <TabsContent value="kody-rabatowe">
                    <div>Zarządzanie kodami rabatowymi</div>
                  </TabsContent>
                  <TabsContent value="vouchery">
                    <div>Zarządzanie voucherami</div>
                  </TabsContent>
                </>
              )}
            </Tabs>
          </main>
        </div>

        {/* Payment Modal */}
        <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
          <DialogContent className="sm:max-w-md">
            <DialogHeader>
              <DialogTitle>Wybierz formę płatności</DialogTitle>
              <DialogDescription>
                Po wybraniu metody zostaniesz przekierowany do płatności.
              </DialogDescription>
            </DialogHeader>
            <RadioGroup value={selectedPayment} onValueChange={setSelectedPayment} className="space-y-4 pt-4">
              <Label className="flex items-center p-4 rounded-lg border-2 cursor-pointer transition-colors has-[:checked]:border-primary has-[:checked]:bg-muted">
                <RadioGroupItem value="blik" id="pay-blik" className="hidden" />
                <span className="font-semibold">BLIK / Przelew</span>
              </Label>
              <Label className="flex items-center p-4 rounded-lg border-2 cursor-pointer transition-colors has-[:checked]:border-primary has-[:checked]:bg-muted">
                <RadioGroupItem value="paypal" id="pay-paypal" className="hidden" />
                <span className="font-semibold">PayPal</span>
              </Label>
              <Label className="flex items-center p-4 rounded-lg border-2 cursor-pointer transition-colors has-[:checked]:border-primary has-[:checked]:bg-muted">
                <RadioGroupItem value="psc" id="pay-psc" className="hidden" />
                <span className="font-semibold">Skiny CS2 / Paysafecard</span>
              </Label>
            </RadioGroup>
            <DialogFooter className="sm:justify-end">
              <Button type="button">
                Zapłać {topUpAmount} PLN
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </>
    );
  }

  return null;
}