'use client'

import { SessionProvider } from 'next-auth/react'
import { useState, createContext, useContext, ReactNode } from 'react'
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"
import { ThemeProvider } from 'next-themes'
import { Button } from '@/components/ui/button'

type AlertDialogContextType = {
  showAlert: (title: string, description: string) => void;
  showConfirm: (title: string, description: string, onConfirm: () => void) => void;
};

const AlertDialogContext = createContext<AlertDialogContextType | undefined>(undefined);

export function useAlertDialog() {
  const context = useContext(AlertDialogContext);
  if (!context) {
    throw new Error('useAlertDialog must be used within an AlertDialogProvider');
  }
  return context;
}

export function AlertDialogProvider({ children }: { children: ReactNode }) {
  const [dialogState, setDialogState] = useState<{
    isOpen: boolean;
    title: string;
    description: string;
    onConfirm?: () => void;
    isConfirmation: boolean;
  } | null>(null);

  const showAlert = (title: string, description: string) => {
    setDialogState({
      isOpen: true,
      title,
      description,
      isConfirmation: false,
    });
  };

  const showConfirm = (title: string, description: string, onConfirm: () => void) => {
    setDialogState({
      isOpen: true,
      title,
      description,
      onConfirm,
      isConfirmation: true,
    });
  };

  const handleClose = () => {
    setDialogState(null);
  };

  const handleConfirm = () => {
    if (dialogState?.onConfirm) {
      dialogState.onConfirm();
    }
    handleClose();
  };

  return (
    <ThemeProvider attribute="class" defaultTheme="system" enableSystem>
      <AlertDialogContext.Provider value={{ showAlert, showConfirm }}>
        {children}
        {dialogState && (
          <AlertDialog open={dialogState.isOpen} onOpenChange={handleClose}>
            <AlertDialogContent>
              <AlertDialogHeader>
                <AlertDialogTitle>{dialogState.title}</AlertDialogTitle>
                <AlertDialogDescription>
                  {dialogState.description}
                </AlertDialogDescription>
              </AlertDialogHeader>
              <AlertDialogFooter>
                {dialogState.isConfirmation ? (
                  <>
                    <AlertDialogCancel onClick={handleClose}>Anuluj</AlertDialogCancel>
                    <AlertDialogAction onClick={handleConfirm}>Potwierdź</AlertDialogAction>
                  </>
                ) : (
                  <AlertDialogAction onClick={handleClose}>OK</AlertDialogAction>
                )}
              </AlertDialogFooter>
            </AlertDialogContent>
          </AlertDialog>
        )}
      </AlertDialogContext.Provider>
    </ThemeProvider>
  );
}

export default function Providers({ children }: { children: React.ReactNode }) {
  return (
    <SessionProvider>
      <AlertDialogProvider>
        {children}
      </AlertDialogProvider>
    </SessionProvider>
  )
}